FUNCTION Show-FolderBrowserDialog {
	  <#
	  .Synopsis
	  Opens the Folder Browser dialog and allows you to select a local folder.
	  .DESCRIPTION
	  Opens the Folder Browser dialog and allows you to select a local folder. You can specify the root folder and make the new folder button available.
	  .EXAMPLE
	  Opens the folder browser dialog without any options. 
      Opens in the MyDocuments folder by default. And the new folder button is not available.
      $folder = Show-FolderBrowserDialog -Verbose
	    .EXAMPLE
      Opens the folder browser dialog with options. 
      Opens in the MyDocuments folder by default. And the new folder button is available.
      $folder = Show-FolderBrowserDialog -ShowNewFolderButton -Verbose
      .EXAMPLE
      Opens the folder browser dialog with options. 
      Opens in the specified folder. And the new folder button is available.
      $folder = Show-FolderBrowserDialog -RootFolder = 'c:\temp' -ShowNewFolderButton -Verbose
	  #>
    [CmdletBinding()]
    param(
        # Path of folder to open in the dialog. By default it opens in the MyDocuments folder.
        [ValidateNotNullOrEmpty()]
        [ValidateScript({ Test-Path -Path $_ })]
        [Parameter()]
        [string] $RootFolder = [System.Environment+SpecialFolder]::MyDocuments,

        # When included, the New Folder button will be enabled.
        [Parameter()]
        [switch] $ShowNewFolderButton
    ) # end param...

    BEGIN{
        $CmdletName = $MyInvocation.MyCommand.Name
        $StartTime = Get-Date
        Write-Verbose -Message "[BEGIN] $StartTime - Entering '$CmdletName' Function..."

        Add-Type -AssemblyName System.Windows.Forms
    } # end BEGIN...
    
    PROCESS{    
        
        try{
          # Create the FolderBrowserDialog object.
            $FolderBrowserDialog = New-Object System.Windows.Forms.FolderBrowserDialog -Property @{
              Description = "Select desired folder."
              SelectedPath = $RootFolder
              ShowNewFolderButton = $ShowNewFolderButton
            }

            $result = $FolderBrowserDialog.ShowDialog((New-Object System.Windows.Forms.Form -Property @{TopMost = $true; TopLevel = $true}))
            if($result -eq 'OK'){
              # Output the name for the selected folder.
              Write-Verbose -Message "'$($FolderBrowserDialog.SelectedPath)' was selected."
              Write-Output $FolderBrowserDialog.SelectedPath
            } else {
              throw "Folder browser dialog was canceled."
            }

        } catch {
            Write-Warning -Message "[PROCESS] $_"
        }

    } # end PROCESS...

    END {
        # Release all memory for the dialog.
        $FolderBrowserDialog.Dispose()
    
        $EndTime = Get-Date
        Write-Verbose -Message "[END] It took $($EndTime - $StartTime) to complete the process."
        Write-Verbose -Message "[END] $EndTime - Exiting '$CmdletName' Function..."
    
    } # end END...

} #end  Function Show-OpenFileDialog
Export-ModuleMember -Function Show-FolderBrowserDialog